<?php
session_start();
require_once '../config/config.php';
require_once '../includes/functions.php';

header('Content-Type: application/json');

if (!isLoggedIn()) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Oturum geçersiz.']);
    exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Geçersiz istek metodu.']);
    exit;
}

try {
    $input = json_decode(file_get_contents('php://input'), true);
    
    $quote_id = intval($input['quote_id'] ?? 0);
    
    if (!$quote_id) {
        throw new Exception('Geçersiz teklif ID.');
    }
    
    // Teklifi kontrol et
    $stmt = $pdo->prepare("
        SELECT q.*, qr.id as request_id,
               c.first_name, c.last_name, c.email, c.phone,
               v.brand, v.model, v.year
        FROM quotes q
        JOIN quote_requests qr ON q.quote_request_id = qr.id
        JOIN customers c ON qr.customer_id = c.id
        JOIN vehicles v ON qr.vehicle_id = v.id
        WHERE q.id = ?
    ");
    $stmt->execute([$quote_id]);
    $quote = $stmt->fetch();
    
    if (!$quote) {
        throw new Exception('Teklif bulunamadı.');
    }
    
    // E-posta gönderme
    $email_sent = false;
    if (!empty($quote['email'])) {
        try {
            require_once '../includes/email_functions.php';
            
            $subject = "Teklif Tekrar Gönderildi - Başvuru No: #{$quote['request_id']}";
            
            $message = "
            <div style='font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto; background: #ffffff;'>
                <!-- Header -->
                <div style='background: linear-gradient(135deg, #2563eb 0%, #3b82f6 100%); color: white; padding: 30px; text-align: center;'>
                    <h1 style='margin: 0; font-size: 28px; font-weight: bold;'>
                        <i style='margin-right: 10px;'>🚗</i>Araç Bakım Servisi
                    </h1>
                    <p style='margin: 10px 0 0 0; font-size: 16px; opacity: 0.9;'>
                        Teklifiniz tekrar gönderilmiştir
                    </p>
                </div>
                
                <!-- Content -->
                <div style='padding: 30px;'>
                    <div style='background: #dbeafe; border-radius: 12px; padding: 20px; margin-bottom: 25px; text-align: center;'>
                        <h2 style='color: #1e40af; margin: 0 0 10px 0; font-size: 24px;'>
                            📋 Teklifiniz Hazır!
                        </h2>
                        <p style='color: #2563eb; font-size: 18px; font-weight: bold; margin: 0;'>
                            Başvuru No: #{$quote['request_id']}
                        </p>
                    </div>
                    
                    <p style='color: #334155; font-size: 16px; line-height: 1.6; margin-bottom: 25px;'>
                        Sayın <strong>{$quote['first_name']} {$quote['last_name']}</strong>,
                    </p>
                    
                    <p style='color: #334155; font-size: 16px; line-height: 1.6; margin-bottom: 25px;'>
                        {$quote['brand']} {$quote['model']} ({$quote['year']}) aracınız için hazırladığımız teklif 
                        tekrar gönderilmiştir. Teklifinizi görüntülemek ve işlem yapmak için aşağıdaki bağlantıyı kullanabilirsiniz.
                    </p>
                    
                    <div style='text-align: center; margin: 30px 0;'>
                        <a href='{$quote['quote_link']}' 
                           style='background: linear-gradient(135deg, #2563eb 0%, #3b82f6 100%); 
                                  color: white; 
                                  padding: 15px 30px; 
                                  border-radius: 8px; 
                                  text-decoration: none; 
                                  font-weight: bold; 
                                  display: inline-block;'>
                            📋 Teklifi Görüntüle
                        </a>
                    </div>
                    
                    <div style='background: #f8fafc; border-radius: 12px; padding: 20px; margin-bottom: 25px;'>
                        <h3 style='color: #1e40af; margin: 0 0 15px 0; font-size: 18px;'>
                            ⏰ Önemli Bilgiler
                        </h3>
                        <ul style='color: #334155; margin: 0; padding-left: 20px;'>
                            <li style='margin-bottom: 8px;'>Teklif geçerlilik tarihi: " . formatDate($quote['valid_until'], 'd.m.Y H:i') . "</li>
                            <li style='margin-bottom: 8px;'>Toplam tutar: " . formatMoney($quote['total_amount']) . "</li>
                            <li style='margin-bottom: 8px;'>Ödeme seçenekleri mevcuttur</li>
                        </ul>
                    </div>
                    
                    <p style='color: #334155; font-size: 16px; line-height: 1.6; margin-bottom: 25px;'>
                        <strong>📞 İletişim:</strong><br>
                        Sorularınız için bize ulaşabilirsiniz:<br>
                        Telefon: <a href='tel:+905321234567' style='color: #2563eb; text-decoration: none;'>+90 532 123 45 67</a><br>
                        E-posta: <a href='mailto:info@aracbakim.com' style='color: #2563eb; text-decoration: none;'>info@aracbakim.com</a>
                    </p>
                </div>
                
                <!-- Footer -->
                <div style='background: #1e40af; color: white; padding: 20px; text-align: center; font-size: 14px;'>
                    <p style='margin: 0 0 10px 0;'>
                        © 2024 Araç Bakım Servisi - Tüm hakları saklıdır.
                    </p>
                    <p style='margin: 0; opacity: 0.8;'>
                        Bu e-posta otomatik olarak gönderilmiştir.
                    </p>
                </div>
            </div>
            ";
            
            // E-posta fonksiyonlarını dahil et
            require_once '../includes/email_functions.php';
            $email_sent = sendEmail($quote['email'], $subject, $message);
            
        } catch (Exception $e) {
            error_log("Resend quote email error: " . $e->getMessage());
        }
    }
    
    // Audit log
    logAudit('quote_resent', 'quotes', $quote_id, $_SESSION['user_id'], [
        'quote_id' => $quote_id,
        'customer_email' => $quote['email'],
        'email_sent' => $email_sent
    ]);
    
    echo json_encode([
        'success' => true,
        'message' => $email_sent ? 'Teklif başarıyla gönderildi!' : 'Teklif gönderildi ancak e-posta gönderilemedi.',
        'email_sent' => $email_sent
    ]);
    
} catch (Exception $e) {
    error_log("Resend quote error: " . $e->getMessage());
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
}
?>
