<?php
session_start();
require_once '../config/config.php';
require_once '../includes/functions.php';

if (!isLoggedIn()) {
    redirect('login.php');
}

$product_id = intval($_GET['id'] ?? 0);
if (!$product_id) {
    redirect('products.php');
}

// Ürün bilgilerini al
try {
    $stmt = $pdo->prepare("
        SELECT p.*, pc.name as category_name 
        FROM products p 
        LEFT JOIN product_categories pc ON p.category_id = pc.id 
        WHERE p.id = ?
    ");
    $stmt->execute([$product_id]);
    $product = $stmt->fetch();
    
    if (!$product) {
        $_SESSION['error_message'] = 'Ürün bulunamadı.';
        redirect('products.php');
    }
} catch (Exception $e) {
    $_SESSION['error_message'] = 'Bir hata oluştu.';
    redirect('products.php');
}

// Kategorileri al
try {
    $stmt = $pdo->query("SELECT * FROM product_categories WHERE is_active = 1 ORDER BY name");
    $categories = $stmt->fetchAll();
} catch (Exception $e) {
    $categories = [];
}

$success_message = '';
$error_message = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $category_id = intval($_POST['category_id'] ?? 0);
    $name = trim($_POST['name'] ?? '');
    $description = trim($_POST['description'] ?? '');
    $type = $_POST['type'] ?? '';
    $sku = trim($_POST['sku'] ?? '');
    $price = floatval($_POST['price'] ?? 0);
    $cost_price = floatval($_POST['cost_price'] ?? 0);
    $stock_quantity = intval($_POST['stock_quantity'] ?? 0);
    $min_stock_level = intval($_POST['min_stock_level'] ?? 0);
    $estimated_duration = intval($_POST['estimated_duration'] ?? 0);
    $warranty_period = intval($_POST['warranty_period'] ?? 0);
    $is_active = isset($_POST['is_active']);
    
    $errors = [];
    
    if (empty($name)) $errors[] = 'Ürün/hizmet adı gereklidir.';
    if (empty($type)) $errors[] = 'Tür seçimi gereklidir.';
    if ($price <= 0) $errors[] = 'Geçerli bir fiyat giriniz.';
    
    // SKU kontrolü (mevcut ürün hariç)
    if ($sku) {
        $stmt = $pdo->prepare("SELECT id FROM products WHERE sku = ? AND id != ?");
        $stmt->execute([$sku, $product_id]);
        if ($stmt->fetch()) {
            $errors[] = 'Bu SKU başka bir ürün tarafından kullanılıyor.';
        }
    }
    
    if (empty($errors)) {
        try {
            $stmt = $pdo->prepare("
                UPDATE products 
                SET category_id = ?, name = ?, description = ?, type = ?, sku = ?, price = ?, cost_price = ?, 
                    stock_quantity = ?, min_stock_level = ?, estimated_duration = ?, warranty_period = ?, 
                    is_active = ?, updated_at = NOW()
                WHERE id = ?
            ");
            
            $stmt->execute([
                $category_id ?: null, $name, $description, $type, $sku ?: null, $price, $cost_price,
                $stock_quantity, $min_stock_level, $estimated_duration, $warranty_period, 
                $is_active ? 1 : 0, $product_id
            ]);
            
            // Audit log
            logAudit('product_updated', 'products', $product_id, null, [
                'name' => $name,
                'type' => $type,
                'price' => $price,
                'updated_by' => $_SESSION['user_id']
            ]);
            
            $success_message = 'Ürün başarıyla güncellendi.';
            
            // Güncellenmiş verileri yeniden yükle
            $stmt = $pdo->prepare("
                SELECT p.*, pc.name as category_name 
                FROM products p 
                LEFT JOIN product_categories pc ON p.category_id = pc.id 
                WHERE p.id = ?
            ");
            $stmt->execute([$product_id]);
            $product = $stmt->fetch();
            
        } catch (Exception $e) {
            error_log("Product update error: " . $e->getMessage());
            $error_message = 'Ürün güncellenirken hata oluştu: ' . $e->getMessage();
        }
    } else {
        $error_message = implode('<br>', $errors);
    }
}
?>
<?php require_once 'includes/header.php'; ?>

<section class="section">
    <div class="pagetitle">
        <h1>Ürün Düzenle</h1>
        <nav>
            <ol class="breadcrumb">
                <li class="breadcrumb-item"><a href="dashboard.php">Ana Sayfa</a></li>
                <li class="breadcrumb-item"><a href="products.php">Ürünler</a></li>
                <li class="breadcrumb-item active">Ürün Düzenle</li>
            </ol>
        </nav>
    </div>

    <?php if ($success_message): ?>
    <div class="alert alert-success alert-dismissible fade show" role="alert">
        <i class="fas fa-check-circle me-2"></i><?= $success_message ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
    <?php endif; ?>

    <?php if ($error_message): ?>
    <div class="alert alert-danger alert-dismissible fade show" role="alert">
        <i class="fas fa-exclamation-triangle me-2"></i><?= $error_message ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
    <?php endif; ?>

    <div class="row">
        <div class="col-lg-8">
            <div class="nice-card">
                <div class="card-body">
                    <h5 class="card-title text-primary">
                        <i class="fas fa-edit me-2"></i>Ürün/Hizmet Bilgileri
                    </h5>

                    <form method="POST" class="needs-validation" novalidate>
                        <div class="row">
                            <!-- Temel Bilgiler -->
                            <div class="col-md-6 mb-3">
                                <label for="name" class="form-label">Ürün/Hizmet Adı <span class="text-danger">*</span></label>
                                <input type="text" class="nice-form-control" id="name" name="name" 
                                       value="<?= h($product['name']) ?>" required>
                                <div class="invalid-feedback">Ürün/hizmet adı gereklidir.</div>
                            </div>

                            <div class="col-md-6 mb-3">
                                <label for="category_id" class="form-label">Kategori</label>
                                <select class="nice-form-control" id="category_id" name="category_id">
                                    <option value="">Kategori Seçin</option>
                                    <?php foreach ($categories as $category): ?>
                                    <option value="<?= $category['id'] ?>" 
                                            <?= ($product['category_id'] == $category['id']) ? 'selected' : '' ?>>
                                        <?= h($category['name']) ?>
                                    </option>
                                    <?php endforeach; ?>
                                </select>
                                <?php if ($product['category_name']): ?>
                                <small class="form-text text-muted">Mevcut: <?= h($product['category_name']) ?></small>
                                <?php endif; ?>
                            </div>

                            <div class="col-md-6 mb-3">
                                <label for="type" class="form-label">Tür <span class="text-danger">*</span></label>
                                <select class="nice-form-control" id="type" name="type" required>
                                    <option value="">Tür Seçin</option>
                                    <option value="service" <?= ($product['type'] == 'service') ? 'selected' : '' ?>>Hizmet</option>
                                    <option value="part" <?= ($product['type'] == 'part') ? 'selected' : '' ?>>Parça</option>
                                    <option value="package" <?= ($product['type'] == 'package') ? 'selected' : '' ?>>Paket</option>
                                </select>
                                <div class="invalid-feedback">Tür seçimi gereklidir.</div>
                            </div>

                            <div class="col-md-6 mb-3">
                                <label for="sku" class="form-label">SKU/Kod</label>
                                <input type="text" class="nice-form-control" id="sku" name="sku" 
                                       value="<?= h($product['sku']) ?>" placeholder="Ürün kodu (opsiyonel)">
                                <small class="form-text text-muted">Benzersiz ürün kodu</small>
                            </div>

                            <div class="col-12 mb-3">
                                <label for="description" class="form-label">Açıklama</label>
                                <textarea class="nice-form-control" id="description" name="description" rows="4" 
                                          placeholder="Ürün/hizmet açıklaması..."><?= h($product['description']) ?></textarea>
                            </div>

                            <!-- Fiyat Bilgileri -->
                            <div class="col-12 mb-4">
                                <h6 class="text-primary border-bottom pb-2">
                                    <i class="fas fa-money-bill-wave me-2"></i>Fiyat Bilgileri
                                </h6>
                            </div>

                            <div class="col-md-6 mb-3">
                                <label for="price" class="form-label">Satış Fiyatı (₺) <span class="text-danger">*</span></label>
                                <input type="number" class="nice-form-control" id="price" name="price" 
                                       value="<?= $product['price'] ?>" step="0.01" min="0" required>
                                <div class="invalid-feedback">Geçerli bir fiyat giriniz.</div>
                            </div>

                            <div class="col-md-6 mb-3">
                                <label for="cost_price" class="form-label">Maliyet Fiyatı (₺)</label>
                                <input type="number" class="nice-form-control" id="cost_price" name="cost_price" 
                                       value="<?= $product['cost_price'] ?>" step="0.01" min="0">
                                <small class="form-text text-muted">İç maliyet hesaplaması için</small>
                            </div>

                            <!-- Stok Bilgileri -->
                            <div class="col-12 mb-4">
                                <h6 class="text-primary border-bottom pb-2">
                                    <i class="fas fa-boxes me-2"></i>Stok Bilgileri
                                </h6>
                            </div>

                            <div class="col-md-6 mb-3">
                                <label for="stock_quantity" class="form-label">Stok Miktarı</label>
                                <input type="number" class="nice-form-control" id="stock_quantity" name="stock_quantity" 
                                       value="<?= $product['stock_quantity'] ?>" min="0">
                                <?php if ($product['stock_quantity'] <= $product['min_stock_level'] && $product['min_stock_level'] > 0): ?>
                                <small class="text-danger">⚠️ Stok seviyesi minimum seviyenin altında!</small>
                                <?php endif; ?>
                            </div>

                            <div class="col-md-6 mb-3">
                                <label for="min_stock_level" class="form-label">Minimum Stok Seviyesi</label>
                                <input type="number" class="nice-form-control" id="min_stock_level" name="min_stock_level" 
                                       value="<?= $product['min_stock_level'] ?>" min="0">
                                <small class="form-text text-muted">Uyarı için minimum seviye</small>
                            </div>

                            <!-- Hizmet Bilgileri -->
                            <div class="col-12 mb-4">
                                <h6 class="text-primary border-bottom pb-2">
                                    <i class="fas fa-tools me-2"></i>Hizmet Bilgileri
                                </h6>
                            </div>

                            <div class="col-md-6 mb-3">
                                <label for="estimated_duration" class="form-label">Tahmini Süre (dakika)</label>
                                <input type="number" class="nice-form-control" id="estimated_duration" name="estimated_duration" 
                                       value="<?= $product['estimated_duration'] ?>" min="0">
                                <small class="form-text text-muted">Hizmet süresi (dakika cinsinden)</small>
                            </div>

                            <div class="col-md-6 mb-3">
                                <label for="warranty_period" class="form-label">Garanti Süresi (gün)</label>
                                <input type="number" class="nice-form-control" id="warranty_period" name="warranty_period" 
                                       value="<?= $product['warranty_period'] ?>" min="0">
                                <small class="form-text text-muted">Garanti süresi (gün cinsinden)</small>
                            </div>

                            <!-- Durum -->
                            <div class="col-12 mb-4">
                                <div class="form-check form-switch">
                                    <input class="form-check-input" type="checkbox" id="is_active" name="is_active" 
                                           <?= $product['is_active'] ? 'checked' : '' ?>>
                                    <label class="form-check-label" for="is_active">
                                        <strong>Aktif</strong>
                                        <small class="text-muted d-block">Ürün/hizmet aktif olarak kullanılabilir</small>
                                    </label>
                                </div>
                            </div>
                        </div>

                        <div class="d-flex gap-2">
                            <button type="submit" class="nice-btn nice-btn-success">
                                <i class="fas fa-save me-2"></i>Güncelle
                            </button>
                            <a href="products.php" class="nice-btn nice-btn-outline-secondary">
                                <i class="fas fa-arrow-left me-2"></i>Geri Dön
                            </a>
                            <button type="button" class="nice-btn nice-btn-outline-danger ms-auto" onclick="deleteProduct()">
                                <i class="fas fa-trash me-2"></i>Sil
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>

        <!-- Sağ Kolon - Bilgiler ve İstatistikler -->
        <div class="col-lg-4">
            <!-- Ürün Bilgileri -->
            <div class="nice-card mb-4">
                <div class="card-body">
                    <h5 class="card-title text-primary">
                        <i class="fas fa-info-circle me-2"></i>Ürün Bilgileri
                    </h5>
                    
                    <div class="row text-center">
                        <div class="col-6 mb-3">
                            <div class="border rounded p-2">
                                <h6 class="text-primary mb-1">ID</h6>
                                <span class="text-muted"><?= $product['id'] ?></span>
                            </div>
                        </div>
                        <div class="col-6 mb-3">
                            <div class="border rounded p-2">
                                <h6 class="text-info mb-1">Durum</h6>
                                <span class="badge <?= $product['is_active'] ? 'bg-success' : 'bg-danger' ?>">
                                    <?= $product['is_active'] ? 'Aktif' : 'Pasif' ?>
                                </span>
                            </div>
                        </div>
                        <div class="col-12 mb-3">
                            <div class="border rounded p-2">
                                <h6 class="text-warning mb-1">Oluşturulma</h6>
                                <small class="text-muted"><?= formatDate($product['created_at'], 'd.m.Y H:i') ?></small>
                            </div>
                        </div>
                        <?php if ($product['updated_at']): ?>
                        <div class="col-12">
                            <div class="border rounded p-2">
                                <h6 class="text-secondary mb-1">Son Güncelleme</h6>
                                <small class="text-muted"><?= formatDate($product['updated_at'], 'd.m.Y H:i') ?></small>
                            </div>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>

            <!-- Kar Marjı Hesaplama -->
            <?php if ($product['cost_price'] > 0): ?>
            <div class="nice-card mb-4">
                <div class="card-body">
                    <h5 class="card-title text-primary">
                        <i class="fas fa-chart-line me-2"></i>Kar Analizi
                    </h5>
                    
                    <?php
                    $margin = (($product['price'] - $product['cost_price']) / $product['price']) * 100;
                    $profit = $product['price'] - $product['cost_price'];
                    ?>
                    
                    <div class="mb-3">
                        <div class="d-flex justify-content-between">
                            <span>Satış Fiyatı:</span>
                            <strong>₺<?= number_format($product['price'], 2) ?></strong>
                        </div>
                        <div class="d-flex justify-content-between">
                            <span>Maliyet:</span>
                            <span>₺<?= number_format($product['cost_price'], 2) ?></span>
                        </div>
                        <hr>
                        <div class="d-flex justify-content-between">
                            <span>Kar:</span>
                            <strong class="text-success">₺<?= number_format($profit, 2) ?></strong>
                        </div>
                        <div class="d-flex justify-content-between">
                            <span>Kar Marjı:</span>
                            <strong class="text-primary">%<?= number_format($margin, 1) ?></strong>
                        </div>
                    </div>
                    
                    <div class="progress">
                        <div class="progress-bar <?= $margin > 30 ? 'bg-success' : ($margin > 15 ? 'bg-warning' : 'bg-danger') ?>" 
                             style="width: <?= min($margin, 100) ?>%"></div>
                    </div>
                    <small class="text-muted">
                        <?= $margin > 30 ? 'Yüksek kar marjı' : ($margin > 15 ? 'Orta kar marjı' : 'Düşük kar marjı') ?>
                    </small>
                </div>
            </div>
            <?php endif; ?>

            <!-- Yardım -->
            <div class="nice-card">
                <div class="card-body">
                    <h5 class="card-title text-primary">
                        <i class="fas fa-question-circle me-2"></i>Yardım
                    </h5>
                    
                    <div class="alert alert-info">
                        <h6><i class="fas fa-lightbulb me-2"></i>İpuçları</h6>
                        <ul class="mb-0 small">
                            <li>SKU değiştirirken dikkatli olun, stok takibini etkileyebilir</li>
                            <li>Fiyat değişiklikleri mevcut teklifleri etkilemez</li>
                            <li>Pasif ürünler yeni tekliflerde görünmez</li>
                            <li>Stok seviyesi minimum seviyenin altındaysa uyarı alırsınız</li>
                        </ul>
                    </div>
                </div>
            </div>
        </div>
    </div>
</section>

<script>
// Form validation
(function() {
    'use strict';
    window.addEventListener('load', function() {
        var forms = document.getElementsByClassName('needs-validation');
        var validation = Array.prototype.filter.call(forms, function(form) {
            form.addEventListener('submit', function(event) {
                if (form.checkValidity() === false) {
                    event.preventDefault();
                    event.stopPropagation();
                }
                form.classList.add('was-validated');
            }, false);
        });
    }, false);
})();

// Kar marjı hesaplama
function calculateMargin() {
    const price = parseFloat(document.getElementById('price').value) || 0;
    const cost = parseFloat(document.getElementById('cost_price').value) || 0;
    
    if (price > 0 && cost > 0) {
        const margin = ((price - cost) / price * 100).toFixed(2);
        const profit = (price - cost).toFixed(2);
        console.log('Kar: ₺' + profit + ' (Marj: %' + margin + ')');
    }
}

document.getElementById('price').addEventListener('input', calculateMargin);
document.getElementById('cost_price').addEventListener('input', calculateMargin);

// Ürün silme
function deleteProduct() {
    if (confirm('Bu ürünü silmek istediğinizden emin misiniz?\n\nBu işlem geri alınamaz ve ürünü kullanan mevcut teklifler etkilenebilir.')) {
        window.location.href = 'product_delete.php?id=<?= $product_id ?>';
    }
}

// Stok uyarısı
document.getElementById('stock_quantity').addEventListener('input', function() {
    const stock = parseInt(this.value) || 0;
    const minLevel = parseInt(document.getElementById('min_stock_level').value) || 0;
    
    if (stock <= minLevel && minLevel > 0) {
        this.classList.add('border-warning');
        if (!document.querySelector('.stock-warning')) {
            const warning = document.createElement('small');
            warning.className = 'text-warning stock-warning';
            warning.innerHTML = '⚠️ Stok seviyesi minimum seviyenin altında!';
            this.parentNode.appendChild(warning);
        }
    } else {
        this.classList.remove('border-warning');
        const warning = document.querySelector('.stock-warning');
        if (warning) warning.remove();
    }
});
</script>

<?php require_once 'includes/footer.php'; ?>