<?php
session_start();
require_once '../config/config.php';
require_once '../includes/functions.php';

header('Content-Type: application/json');

function jsonResponse($data, $status = 200) {
    http_response_code($status);
    echo json_encode($data);
    exit;
}

if (!isLoggedIn()) {
    jsonResponse(['success' => false, 'message' => 'Oturum geçersiz.'], 401);
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    jsonResponse(['success' => false, 'message' => 'Geçersiz istek metodu.'], 405);
}

try {
    $quote_id = intval($_POST['quote_id'] ?? 0);
    $discount_rate = floatval($_POST['discount_rate'] ?? 0);
    $tax_rate = floatval($_POST['tax_rate'] ?? 18);
    $terms_conditions = trim($_POST['terms_conditions'] ?? '');
    $internal_notes = trim($_POST['internal_notes'] ?? '');
    $items = $_POST['items'] ?? [];
    
    if ($quote_id <= 0) {
        jsonResponse(['success' => false, 'message' => 'Geçersiz teklif ID.']);
    }
    
    if (empty($items)) {
        jsonResponse(['success' => false, 'message' => 'En az bir ürün/hizmet eklemelisiniz.']);
    }
    
    // Teklif bilgilerini al ve kontrol et
    $stmt = $pdo->prepare("
        SELECT q.*, qr.id as request_id
        FROM quotes q
        JOIN quote_requests qr ON q.quote_request_id = qr.id
        WHERE q.id = ?
    ");
    $stmt->execute([$quote_id]);
    $quote = $stmt->fetch();
    
    if (!$quote) {
        jsonResponse(['success' => false, 'message' => 'Teklif bulunamadı.']);
    }
    
    // Düzenlenebilir durumda mı kontrol et
    $editable_statuses = ['DRAFT', 'SENT', 'VIEWED'];
    if (!in_array($quote['status'], $editable_statuses)) {
        jsonResponse(['success' => false, 'message' => 'Bu durumda olan teklif düzenlenemez.']);
    }
    
    $pdo->beginTransaction();
    
    // Toplam tutarları hesapla
    $subtotal = 0;
    $valid_items = [];
    
    foreach ($items as $item) {
        $name = trim($item['name'] ?? '');
        $description = trim($item['description'] ?? '');
        $quantity = floatval($item['quantity'] ?? 1);
        $unit_price = floatval($item['unit_price'] ?? 0);
        $product_id = !empty($item['product_id']) ? intval($item['product_id']) : null;
        
        if (!$name || $quantity <= 0 || $unit_price < 0) {
            continue;
        }
        
        $line_total = $quantity * $unit_price;
        $subtotal += $line_total;
        
        $valid_items[] = [
            'product_id' => $product_id,
            'name' => $name,
            'description' => $description,
            'quantity' => $quantity,
            'unit_price' => $unit_price,
            'line_total' => $line_total
        ];
    }
    
    if (empty($valid_items)) {
        jsonResponse(['success' => false, 'message' => 'Geçerli ürün/hizmet bulunamadı.']);
    }
    
    // İndirim ve vergi hesapla
    $discount_amount = $subtotal * ($discount_rate / 100);
    $after_discount = $subtotal - $discount_amount;
    $tax_amount = $after_discount * ($tax_rate / 100);
    $total_amount = $after_discount + $tax_amount;
    
    // Geçerlilik tarihi (48 saat sonra)
    $valid_until = date('Y-m-d H:i:s', strtotime('+48 hours'));
    
    // Teklifi güncelle
    $stmt = $pdo->prepare("
        UPDATE quotes 
        SET subtotal = ?, discount_percentage = ?, discount_amount = ?, 
            tax_rate = ?, tax_amount = ?, total_amount = ?, 
            valid_until = ?, terms_conditions = ?, internal_notes = ?,
            updated_at = NOW()
        WHERE id = ?
    ");
    $stmt->execute([
        $subtotal, $discount_rate, $discount_amount,
        $tax_rate / 100, $tax_amount, $total_amount,
        $valid_until, $terms_conditions, $internal_notes, $quote_id
    ]);
    
    // Mevcut teklif kalemlerini sil
    $stmt = $pdo->prepare("DELETE FROM quote_items WHERE quote_id = ?");
    $stmt->execute([$quote_id]);
    
    // Yeni teklif kalemlerini ekle
    $sort_order = 0;
    foreach ($valid_items as $item) {
        $stmt = $pdo->prepare("
            INSERT INTO quote_items (quote_id, product_id, description, quantity, unit_price,
                                   discount_amount, tax_rate, line_total, sort_order)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
        ");
        $stmt->execute([
            $quote_id, $item['product_id'], $item['name'] ?: $item['description'], $item['quantity'], $item['unit_price'],
            0, $tax_rate / 100, $item['line_total'], ++$sort_order
        ]);
    }
    
    // Audit log
    logAudit('quote_updated', 'quotes', $quote_id, null, [
        'quote_number' => $quote['quote_number'],
        'total_amount' => $total_amount,
        'item_count' => count($valid_items),
        'updated_by' => $_SESSION['user_id']
    ]);
    
    $pdo->commit();
    
    jsonResponse([
        'success' => true,
        'message' => 'Teklif başarıyla güncellendi.',
        'quote_id' => $quote_id,
        'quote_number' => $quote['quote_number'],
        'total_amount' => $total_amount
    ]);
    
} catch (Exception $e) {
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    
    error_log("Edit quote error: " . $e->getMessage());
    jsonResponse(['success' => false, 'message' => $e->getMessage()], 500);
}
?>
