<?php
session_start();
require_once '../config/config.php';
require_once '../includes/functions.php';
require_once '../includes/email_functions.php';

$pageTitle = 'E-posta Kurulumu';
$currentPage = 'email_setup';

if (!isLoggedIn()) {
    redirect('login.php');
}

$message = '';
$messageType = '';

// SMTP ayarlarını kaydet
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['save_smtp'])) {
    $smtp_host = trim($_POST['smtp_host'] ?? '');
    $smtp_port = trim($_POST['smtp_port'] ?? '587');
    $smtp_username = trim($_POST['smtp_username'] ?? '');
    $smtp_password = trim($_POST['smtp_password'] ?? '');
    $admin_email = trim($_POST['admin_email'] ?? '');
    
    if (empty($smtp_host) || empty($smtp_username) || empty($smtp_password) || empty($admin_email)) {
        $message = 'Lütfen tüm alanları doldurun.';
        $messageType = 'danger';
    } elseif (!filter_var($admin_email, FILTER_VALIDATE_EMAIL)) {
        $message = 'Geçerli bir admin e-posta adresi girin.';
        $messageType = 'danger';
    } else {
        // Config dosyasını güncelle
        $config_path = '../config/config.php';
        $config_content = file_get_contents($config_path);
        
        // SMTP ayarlarını güncelle
        $config_content = preg_replace(
            "/define\('SMTP_HOST', '[^']*'\);/",
            "define('SMTP_HOST', '$smtp_host');",
            $config_content
        );
        $config_content = preg_replace(
            "/define\('SMTP_PORT', [^)]*\);/",
            "define('SMTP_PORT', $smtp_port);",
            $config_content
        );
        $config_content = preg_replace(
            "/define\('SMTP_USERNAME', '[^']*'\);/",
            "define('SMTP_USERNAME', '$smtp_username');",
            $config_content
        );
        $config_content = preg_replace(
            "/define\('SMTP_PASSWORD', '[^']*'\);/",
            "define('SMTP_PASSWORD', '$smtp_password');",
            $config_content
        );
        $config_content = preg_replace(
            "/define\('ADMIN_EMAIL', '[^']*'\);/",
            "define('ADMIN_EMAIL', '$admin_email');",
            $config_content
        );
        
        if (file_put_contents($config_path, $config_content)) {
            $message = 'SMTP ayarları başarıyla kaydedildi! Sayfayı yenileyin.';
            $messageType = 'success';
            
            // Audit log
            logAudit('smtp_settings_updated', 'system', null, null, [
                'smtp_host' => $smtp_host,
                'smtp_username' => $smtp_username,
                'admin_email' => $admin_email,
                'updated_by' => $_SESSION['user_id']
            ]);
        } else {
            $message = 'Config dosyası güncellenemedi. Dosya izinlerini kontrol edin.';
            $messageType = 'danger';
        }
    }
}

// Test e-postası gönder
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['send_test'])) {
    $test_email = trim($_POST['test_email'] ?? '');
    
    if (empty($test_email)) {
        $message = 'Lütfen test e-posta adresi girin.';
        $messageType = 'danger';
    } elseif (!filter_var($test_email, FILTER_VALIDATE_EMAIL)) {
        $message = 'Geçerli bir e-posta adresi girin.';
        $messageType = 'danger';
    } else {
        try {
            $result = sendTestEmail($test_email);
            
            if ($result) {
                $message = "Test e-postası başarıyla gönderildi: $test_email";
                $messageType = 'success';
            } else {
                $message = 'E-posta gönderilemedi. SMTP ayarlarınızı kontrol edin.';
                $messageType = 'danger';
            }
        } catch (Exception $e) {
            $message = 'Hata: ' . $e->getMessage();
            $messageType = 'danger';
            error_log("Test email error: " . $e->getMessage());
        }
    }
}

// Mevcut ayarları al
$current_smtp_host = defined('SMTP_HOST') ? SMTP_HOST : 'smtp.gmail.com';
$current_smtp_port = defined('SMTP_PORT') ? SMTP_PORT : 587;
$current_smtp_username = defined('SMTP_USERNAME') ? SMTP_USERNAME : '';
$current_smtp_password = defined('SMTP_PASSWORD') ? SMTP_PASSWORD : '';
$current_admin_email = defined('ADMIN_EMAIL') ? ADMIN_EMAIL : '';

// SMTP yapılandırma durumu
$smtp_configured = $current_smtp_host !== 'smtp.gmail.com' && 
                  $current_smtp_username !== 'your_email@gmail.com' && 
                  $current_smtp_password !== 'your_app_password' &&
                  !empty($current_smtp_username) && !empty($current_smtp_password);

include 'includes/header.php';
?>

<div class="container-fluid">
    <div class="row">
        <div class="col-12">
            <div class="nice-card">
                <div class="card-header">
                    <h5 class="card-title mb-0">
                        <i class="fas fa-cogs text-primary me-2"></i>
                        E-posta Sistem Kurulumu
                    </h5>
                </div>
                
                <div class="card-body">
                    <?php if ($message): ?>
                    <div class="alert alert-<?= $messageType ?> alert-dismissible fade show" role="alert">
                        <?= htmlspecialchars($message) ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                    <?php endif; ?>
                    
                    <!-- Sistem Durumu -->
                    <div class="row mb-4">
                        <div class="col-12">
                            <div class="nice-card bg-<?= $smtp_configured ? 'success' : 'warning' ?> text-white">
                                <div class="card-body text-center">
                                    <i class="fas fa-<?= $smtp_configured ? 'check-circle' : 'exclamation-triangle' ?> fa-3x mb-3"></i>
                                    <h4><?= $smtp_configured ? 'E-posta Sistemi Aktif' : 'E-posta Sistemi Yapılandırılmamış' ?></h4>
                                    <p class="mb-0">
                                        <?= $smtp_configured ? 'SMTP ayarları yapılandırılmış ve e-posta gönderimi hazır.' : 'Lütfen aşağıdaki formu kullanarak SMTP ayarlarını yapılandırın.' ?>
                                    </p>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- SMTP Yapılandırma Formu -->
                    <div class="row">
                        <div class="col-md-8">
                            <div class="nice-card">
                                <div class="card-header">
                                    <h6 class="card-title mb-0">
                                        <i class="fas fa-server text-info me-2"></i>
                                        SMTP Ayarları
                                    </h6>
                                </div>
                                <div class="card-body">
                                    <form method="POST" class="needs-validation" novalidate>
                                        <div class="row">
                                            <div class="col-md-8">
                                                <div class="mb-3">
                                                    <label for="smtp_host" class="form-label">SMTP Host</label>
                                                    <select class="nice-form-control" id="smtp_host" name="smtp_host" onchange="updateSMTPSettings()">
                                                        <option value="smtp.gmail.com" <?= $current_smtp_host === 'smtp.gmail.com' ? 'selected' : '' ?>>Gmail (smtp.gmail.com)</option>
                                                        <option value="smtp-mail.outlook.com" <?= $current_smtp_host === 'smtp-mail.outlook.com' ? 'selected' : '' ?>>Outlook (smtp-mail.outlook.com)</option>
                                                        <option value="smtp.mail.yahoo.com" <?= $current_smtp_host === 'smtp.mail.yahoo.com' ? 'selected' : '' ?>>Yahoo (smtp.mail.yahoo.com)</option>
                                                        <option value="smtp.yandex.com" <?= $current_smtp_host === 'smtp.yandex.com' ? 'selected' : '' ?>>Yandex (smtp.yandex.com)</option>
                                                        <option value="custom">Özel SMTP Sunucusu</option>
                                                    </select>
                                                </div>
                                            </div>
                                            <div class="col-md-4">
                                                <div class="mb-3">
                                                    <label for="smtp_port" class="form-label">SMTP Port</label>
                                                    <input type="number" class="nice-form-control" id="smtp_port" name="smtp_port" 
                                                           value="<?= $current_smtp_port ?>" required>
                                                </div>
                                            </div>
                                        </div>
                                        
                                        <div id="custom_host_input" style="display: none;">
                                            <div class="mb-3">
                                                <label for="custom_smtp_host" class="form-label">Özel SMTP Host</label>
                                                <input type="text" class="nice-form-control" id="custom_smtp_host" 
                                                       placeholder="mail.example.com">
                                            </div>
                                        </div>
                                        
                                        <div class="mb-3">
                                            <label for="smtp_username" class="form-label">E-posta Adresi</label>
                                            <input type="email" class="nice-form-control" id="smtp_username" name="smtp_username" 
                                                   value="<?= htmlspecialchars($current_smtp_username) ?>" required>
                                            <div class="form-text">Gönderen e-posta adresi</div>
                                        </div>
                                        
                                        <div class="mb-3">
                                            <label for="smtp_password" class="form-label">E-posta Şifresi / App Password</label>
                                            <div class="input-group">
                                                <input type="password" class="nice-form-control" id="smtp_password" name="smtp_password" 
                                                       value="<?= htmlspecialchars($current_smtp_password) ?>" required>
                                                <button type="button" class="nice-btn nice-btn-outline-secondary" onclick="togglePassword()">
                                                    <i class="fas fa-eye" id="password-icon"></i>
                                                </button>
                                            </div>
                                            <div class="form-text">Gmail için App Password kullanın</div>
                                        </div>
                                        
                                        <div class="mb-3">
                                            <label for="admin_email" class="form-label">Admin E-posta Adresi</label>
                                            <input type="email" class="nice-form-control" id="admin_email" name="admin_email" 
                                                   value="<?= htmlspecialchars($current_admin_email) ?>" required>
                                            <div class="form-text">İletişim formlarının gönderileceği adres</div>
                                        </div>
                                        
                                        <div class="d-grid gap-2 d-md-flex">
                                            <button type="submit" name="save_smtp" class="nice-btn nice-btn-primary">
                                                <i class="fas fa-save me-2"></i>
                                                SMTP Ayarlarını Kaydet
                                            </button>
                                        </div>
                                    </form>
                                </div>
                            </div>
                        </div>
                        
                        <div class="col-md-4">
                            <!-- Gmail Kurulum Rehberi -->
                            <div class="nice-card">
                                <div class="card-header">
                                    <h6 class="card-title mb-0">
                                        <i class="fab fa-google text-danger me-2"></i>
                                        Gmail Kurulumu
                                    </h6>
                                </div>
                                <div class="card-body">
                                    <ol class="small">
                                        <li>Google hesabınızda <strong>2FA</strong> aktif edin</li>
                                        <li><a href="https://myaccount.google.com/apppasswords" target="_blank">App Passwords</a> sayfasına gidin</li>
                                        <li>"Mail" seçeneğini seçin</li>
                                        <li>16 karakterlik şifreyi kopyalayın</li>
                                        <li>Bu şifreyi "E-posta Şifresi" alanına yapıştırın</li>
                                    </ol>
                                    
                                    <div class="alert alert-info py-2 mt-3">
                                        <small>
                                            <i class="fas fa-info-circle me-1"></i>
                                            Gmail normal şifrenizi değil, App Password kullanın!
                                        </small>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- Test E-postası -->
                            <?php if ($smtp_configured): ?>
                            <div class="nice-card mt-3">
                                <div class="card-header">
                                    <h6 class="card-title mb-0">
                                        <i class="fas fa-paper-plane text-success me-2"></i>
                                        Test E-postası
                                    </h6>
                                </div>
                                <div class="card-body">
                                    <form method="POST">
                                        <div class="mb-3">
                                            <input type="email" class="nice-form-control" name="test_email" 
                                                   placeholder="test@example.com" required>
                                        </div>
                                        <div class="d-grid">
                                            <button type="submit" name="send_test" class="nice-btn nice-btn-success">
                                                <i class="fas fa-paper-plane me-2"></i>
                                                Test Gönder
                                            </button>
                                        </div>
                                    </form>
                                </div>
                            </div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
function updateSMTPSettings() {
    const hostSelect = document.getElementById('smtp_host');
    const portInput = document.getElementById('smtp_port');
    const customHostDiv = document.getElementById('custom_host_input');
    const customHostInput = document.getElementById('custom_smtp_host');
    
    if (hostSelect.value === 'custom') {
        customHostDiv.style.display = 'block';
        customHostInput.required = true;
    } else {
        customHostDiv.style.display = 'none';
        customHostInput.required = false;
        
        // Port ayarlarını güncelle
        switch(hostSelect.value) {
            case 'smtp.gmail.com':
            case 'smtp-mail.outlook.com':
            case 'smtp.mail.yahoo.com':
            case 'smtp.yandex.com':
                portInput.value = 587;
                break;
            default:
                portInput.value = 587;
        }
    }
}

function togglePassword() {
    const passwordInput = document.getElementById('smtp_password');
    const passwordIcon = document.getElementById('password-icon');
    
    if (passwordInput.type === 'password') {
        passwordInput.type = 'text';
        passwordIcon.className = 'fas fa-eye-slash';
    } else {
        passwordInput.type = 'password';
        passwordIcon.className = 'fas fa-eye';
    }
}

// Form validation
(function() {
    'use strict';
    window.addEventListener('load', function() {
        var forms = document.getElementsByClassName('needs-validation');
        var validation = Array.prototype.filter.call(forms, function(form) {
            form.addEventListener('submit', function(event) {
                if (form.checkValidity() === false) {
                    event.preventDefault();
                    event.stopPropagation();
                }
                form.classList.add('was-validated');
            }, false);
        });
    }, false);
})();

// Sayfa yüklendiğinde custom host kontrolü
document.addEventListener('DOMContentLoaded', function() {
    updateSMTPSettings();
});
</script>

<?php include 'includes/footer.php'; ?>
