<?php
session_start();
require_once '../config/config.php';
require_once '../includes/functions.php';

$pageTitle = 'Kategoriler';
$currentPage = 'categories';

if (!isLoggedIn()) {
    redirect('login.php');
}

// Sayfalama
$page = intval($_GET['page'] ?? 1);
$limit = 20;
$offset = ($page - 1) * $limit;

// Filtreleme
$search = trim($_GET['search'] ?? '');
$status_filter = $_GET['status'] ?? '';

// Kategorileri al
$where_conditions = ['1=1'];
$params = [];

if ($search) {
    $where_conditions[] = "(name LIKE ? OR description LIKE ?)";
    $params[] = "%$search%";
    $params[] = "%$search%";
}

if ($status_filter !== '') {
    $where_conditions[] = "is_active = ?";
    $params[] = intval($status_filter);
}

$where_clause = implode(' AND ', $where_conditions);

// Toplam kayıt sayısı
$count_sql = "SELECT COUNT(*) as total FROM product_categories WHERE $where_clause";
$stmt = $pdo->prepare($count_sql);
$stmt->execute($params);
$total_records = $stmt->fetch()['total'];

$total_pages = ceil($total_records / $limit);

// Kategorileri getir
$sql = "SELECT pc.*, 
               (SELECT COUNT(*) FROM products p WHERE p.category_id = pc.id AND p.is_active = 1) as product_count
        FROM product_categories pc 
        WHERE $where_clause 
        ORDER BY pc.sort_order ASC, pc.name ASC 
        LIMIT $limit OFFSET $offset";

$stmt = $pdo->prepare($sql);
$stmt->execute($params);
$categories = $stmt->fetchAll();

include 'includes/header.php';
?>

<div class="container-fluid">
    <div class="row">
        <div class="col-12">
            <div class="nice-card">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h5 class="card-title mb-0">
                        <i class="fas fa-tags text-primary me-2"></i>
                        Kategori Yönetimi
                    </h5>
                    <button type="button" class="nice-btn nice-btn-primary" data-bs-toggle="modal" data-bs-target="#addCategoryModal">
                        <i class="fas fa-plus"></i> Yeni Kategori
                    </button>
                </div>
                
                <div class="card-body">
                    <!-- Filtreler -->
                    <div class="row mb-4">
                        <div class="col-md-4">
                            <div class="input-group">
                                <span class="input-group-text">
                                    <i class="fas fa-search"></i>
                                </span>
                                <input type="text" class="nice-form-control" id="searchInput" 
                                       placeholder="Kategori ara..." value="<?= h($search) ?>">
                            </div>
                        </div>
                        <div class="col-md-3">
                            <select class="nice-form-control" id="statusFilter">
                                <option value="">Tüm Durumlar</option>
                                <option value="1" <?= $status_filter === '1' ? 'selected' : '' ?>>Aktif</option>
                                <option value="0" <?= $status_filter === '0' ? 'selected' : '' ?>>Pasif</option>
                            </select>
                        </div>
                        <div class="col-md-2">
                            <button type="button" class="nice-btn nice-btn-outline-primary w-100" onclick="applyFilters()">
                                <i class="fas fa-filter"></i> Filtrele
                            </button>
                        </div>
                        <div class="col-md-2">
                            <button type="button" class="nice-btn nice-btn-outline-secondary w-100" onclick="clearFilters()">
                                <i class="fas fa-times"></i> Temizle
                            </button>
                        </div>
                    </div>

                    <!-- İstatistikler -->
                    <div class="row mb-4">
                        <div class="col-md-3">
                            <div class="nice-stat-card bg-primary">
                                <div class="stat-icon">
                                    <i class="fas fa-tags"></i>
                                </div>
                                <div class="stat-content">
                                    <h3><?= $total_records ?></h3>
                                    <p>Toplam Kategori</p>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-3">
                            <div class="nice-stat-card bg-success">
                                <div class="stat-icon">
                                    <i class="fas fa-check-circle"></i>
                                </div>
                                <div class="stat-content">
                                    <?php
                                    $stmt = $pdo->query("SELECT COUNT(*) as count FROM product_categories WHERE is_active = 1");
                                    $active_count = $stmt->fetch()['count'];
                                    ?>
                                    <h3><?= $active_count ?></h3>
                                    <p>Aktif Kategori</p>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-3">
                            <div class="nice-stat-card bg-warning">
                                <div class="stat-icon">
                                    <i class="fas fa-pause-circle"></i>
                                </div>
                                <div class="stat-content">
                                    <?php
                                    $stmt = $pdo->query("SELECT COUNT(*) as count FROM product_categories WHERE is_active = 0");
                                    $inactive_count = $stmt->fetch()['count'];
                                    ?>
                                    <h3><?= $inactive_count ?></h3>
                                    <p>Pasif Kategori</p>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-3">
                            <div class="nice-stat-card bg-info">
                                <div class="stat-icon">
                                    <i class="fas fa-boxes"></i>
                                </div>
                                <div class="stat-content">
                                    <?php
                                    $stmt = $pdo->query("SELECT COUNT(*) as count FROM products WHERE is_active = 1");
                                    $total_products = $stmt->fetch()['count'];
                                    ?>
                                    <h3><?= $total_products ?></h3>
                                    <p>Toplam Ürün</p>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Kategori Tablosu -->
                    <div class="table-responsive">
                        <table class="nice-table">
                            <thead>
                                <tr>
                                    <th>Sıra</th>
                                    <th>Kategori Adı</th>
                                    <th>Açıklama</th>
                                    <th>Ürün Sayısı</th>
                                    <th>Durum</th>
                                    <th>Oluşturma</th>
                                    <th>İşlemler</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (empty($categories)): ?>
                                <tr>
                                    <td colspan="7" class="text-center py-4">
                                        <div class="text-muted">
                                            <i class="fas fa-inbox fa-3x mb-3"></i>
                                            <p>Henüz kategori bulunmuyor.</p>
                                        </div>
                                    </td>
                                </tr>
                                <?php else: ?>
                                <?php foreach ($categories as $category): ?>
                                <tr>
                                    <td>
                                        <span class="nice-badge nice-badge-secondary">
                                            <?= $category['sort_order'] ?>
                                        </span>
                                    </td>
                                    <td>
                                        <div class="d-flex align-items-center">
                                            <div class="category-icon me-2">
                                                <i class="fas fa-tag text-primary"></i>
                                            </div>
                                            <div>
                                                <strong><?= h($category['name']) ?></strong>
                                            </div>
                                        </div>
                                    </td>
                                    <td>
                                        <span class="text-muted">
                                            <?= h($category['description'] ?: 'Açıklama yok') ?>
                                        </span>
                                    </td>
                                    <td>
                                        <span class="nice-badge nice-badge-info">
                                            <?= $category['product_count'] ?> ürün
                                        </span>
                                    </td>
                                    <td>
                                        <?php if ($category['is_active']): ?>
                                            <span class="nice-badge nice-badge-success">
                                                <i class="fas fa-check-circle"></i> Aktif
                                            </span>
                                        <?php else: ?>
                                            <span class="nice-badge nice-badge-warning">
                                                <i class="fas fa-pause-circle"></i> Pasif
                                            </span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <small class="text-muted">
                                            <?= date('d.m.Y H:i', strtotime($category['created_at'])) ?>
                                        </small>
                                    </td>
                                    <td>
                                        <div class="dropdown">
                                            <button class="nice-btn nice-btn-sm nice-btn-outline-primary dropdown-toggle" 
                                                    type="button" data-bs-toggle="dropdown">
                                                <i class="fas fa-cog"></i> İşlemler
                                            </button>
                                            <ul class="dropdown-menu">
                                                <li>
                                                    <a class="dropdown-item" href="#" onclick="editCategory(<?= $category['id'] ?>)">
                                                        <i class="fas fa-edit text-primary"></i> Düzenle
                                                    </a>
                                                </li>
                                                <li>
                                                    <a class="dropdown-item" href="#" onclick="toggleCategoryStatus(<?= $category['id'] ?>, <?= $category['is_active'] ? 0 : 1 ?>)">
                                                        <?php if ($category['is_active']): ?>
                                                            <i class="fas fa-pause text-warning"></i> Pasif Yap
                                                        <?php else: ?>
                                                            <i class="fas fa-play text-success"></i> Aktif Yap
                                                        <?php endif; ?>
                                                    </a>
                                                </li>
                                                <?php if ($category['product_count'] == 0): ?>
                                                <li><hr class="dropdown-divider"></li>
                                                <li>
                                                    <a class="dropdown-item text-danger" href="#" onclick="deleteCategory(<?= $category['id'] ?>)">
                                                        <i class="fas fa-trash"></i> Sil
                                                    </a>
                                                </li>
                                                <?php endif; ?>
                                            </ul>
                                        </div>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>

                    <!-- Sayfalama -->
                    <?php if ($total_pages > 1): ?>
                    <nav aria-label="Sayfa navigasyonu" class="mt-4">
                        <ul class="pagination justify-content-center">
                            <?php if ($page > 1): ?>
                            <li class="page-item">
                                <a class="page-link" href="?page=<?= $page - 1 ?>&search=<?= urlencode($search) ?>&status=<?= urlencode($status_filter) ?>">
                                    <i class="fas fa-chevron-left"></i>
                                </a>
                            </li>
                            <?php endif; ?>
                            
                            <?php for ($i = max(1, $page - 2); $i <= min($total_pages, $page + 2); $i++): ?>
                            <li class="page-item <?= $i == $page ? 'active' : '' ?>">
                                <a class="page-link" href="?page=<?= $i ?>&search=<?= urlencode($search) ?>&status=<?= urlencode($status_filter) ?>">
                                    <?= $i ?>
                                </a>
                            </li>
                            <?php endfor; ?>
                            
                            <?php if ($page < $total_pages): ?>
                            <li class="page-item">
                                <a class="page-link" href="?page=<?= $page + 1 ?>&search=<?= urlencode($search) ?>&status=<?= urlencode($status_filter) ?>">
                                    <i class="fas fa-chevron-right"></i>
                                </a>
                            </li>
                            <?php endif; ?>
                        </ul>
                    </nav>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Kategori Ekleme Modal -->
<div class="modal fade" id="addCategoryModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">
                    <i class="fas fa-plus text-primary me-2"></i>
                    Yeni Kategori Ekle
                </h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form id="addCategoryForm">
                <div class="modal-body">
                    <div class="mb-3">
                        <label for="add_name" class="form-label">Kategori Adı *</label>
                        <input type="text" class="nice-form-control" id="add_name" name="name" required>
                    </div>
                    <div class="mb-3">
                        <label for="add_description" class="form-label">Açıklama</label>
                        <textarea class="nice-form-control" id="add_description" name="description" rows="3"></textarea>
                    </div>
                    <div class="row">
                        <div class="col-md-6">
                            <label for="add_sort_order" class="form-label">Sıra No</label>
                            <input type="number" class="nice-form-control" id="add_sort_order" name="sort_order" value="0" min="0">
                        </div>
                        <div class="col-md-6">
                            <label for="add_is_active" class="form-label">Durum</label>
                            <select class="nice-form-control" id="add_is_active" name="is_active">
                                <option value="1">Aktif</option>
                                <option value="0">Pasif</option>
                            </select>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="nice-btn nice-btn-secondary" data-bs-dismiss="modal">İptal</button>
                    <button type="submit" class="nice-btn nice-btn-primary">
                        <i class="fas fa-save"></i> Kaydet
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Kategori Düzenleme Modal -->
<div class="modal fade" id="editCategoryModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">
                    <i class="fas fa-edit text-primary me-2"></i>
                    Kategori Düzenle
                </h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form id="editCategoryForm">
                <input type="hidden" id="edit_id" name="id">
                <div class="modal-body">
                    <div class="mb-3">
                        <label for="edit_name" class="form-label">Kategori Adı *</label>
                        <input type="text" class="nice-form-control" id="edit_name" name="name" required>
                    </div>
                    <div class="mb-3">
                        <label for="edit_description" class="form-label">Açıklama</label>
                        <textarea class="nice-form-control" id="edit_description" name="description" rows="3"></textarea>
                    </div>
                    <div class="row">
                        <div class="col-md-6">
                            <label for="edit_sort_order" class="form-label">Sıra No</label>
                            <input type="number" class="nice-form-control" id="edit_sort_order" name="sort_order" min="0">
                        </div>
                        <div class="col-md-6">
                            <label for="edit_is_active" class="form-label">Durum</label>
                            <select class="nice-form-control" id="edit_is_active" name="is_active">
                                <option value="1">Aktif</option>
                                <option value="0">Pasif</option>
                            </select>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="nice-btn nice-btn-secondary" data-bs-dismiss="modal">İptal</button>
                    <button type="submit" class="nice-btn nice-btn-primary">
                        <i class="fas fa-save"></i> Güncelle
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
// Filtreleme
function applyFilters() {
    const search = document.getElementById('searchInput').value;
    const status = document.getElementById('statusFilter').value;
    
    const params = new URLSearchParams();
    if (search) params.append('search', search);
    if (status !== '') params.append('status', status);
    
    window.location.href = 'categories.php?' + params.toString();
}

function clearFilters() {
    window.location.href = 'categories.php';
}

// Enter tuşu ile arama
document.getElementById('searchInput').addEventListener('keypress', function(e) {
    if (e.key === 'Enter') {
        applyFilters();
    }
});

// Kategori ekleme
document.getElementById('addCategoryForm').addEventListener('submit', function(e) {
    e.preventDefault();
    
    const formData = new FormData(this);
    formData.append('action', 'add');
    
    fetch('category_actions.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showNotification('Kategori başarıyla eklendi!', 'success');
            setTimeout(() => {
                window.location.reload();
            }, 1000);
        } else {
            showNotification('Hata: ' + data.message, 'error');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showNotification('Bir hata oluştu!', 'error');
    });
});

// Kategori düzenleme
function editCategory(id) {
    fetch(`category_actions.php?action=get&id=${id}`)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                const category = data.category;
                document.getElementById('edit_id').value = category.id;
                document.getElementById('edit_name').value = category.name;
                document.getElementById('edit_description').value = category.description || '';
                document.getElementById('edit_sort_order').value = category.sort_order;
                document.getElementById('edit_is_active').value = category.is_active;
                
                new bootstrap.Modal(document.getElementById('editCategoryModal')).show();
            } else {
                showNotification('Kategori bilgileri alınamadı!', 'error');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            showNotification('Bir hata oluştu!', 'error');
        });
}

// Kategori güncelleme
document.getElementById('editCategoryForm').addEventListener('submit', function(e) {
    e.preventDefault();
    
    const formData = new FormData(this);
    formData.append('action', 'update');
    
    fetch('category_actions.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showNotification('Kategori başarıyla güncellendi!', 'success');
            setTimeout(() => {
                window.location.reload();
            }, 1000);
        } else {
            showNotification('Hata: ' + data.message, 'error');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showNotification('Bir hata oluştu!', 'error');
    });
});

// Kategori durumu değiştirme
function toggleCategoryStatus(id, status) {
    const action = status ? 'activate' : 'deactivate';
    const message = status ? 'aktif' : 'pasif';
    
    if (confirm(`Bu kategoriyi ${message} yapmak istediğinizden emin misiniz?`)) {
        const formData = new FormData();
        formData.append('action', 'toggle_status');
        formData.append('id', id);
        formData.append('status', status);
        
        fetch('category_actions.php', {
            method: 'POST',
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showNotification(`Kategori başarıyla ${message} yapıldı!`, 'success');
                setTimeout(() => {
                    window.location.reload();
                }, 1000);
            } else {
                showNotification('Hata: ' + data.message, 'error');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            showNotification('Bir hata oluştu!', 'error');
        });
    }
}

// Kategori silme
function deleteCategory(id) {
    if (confirm('Bu kategoriyi silmek istediğinizden emin misiniz?\n\nBu işlem geri alınamaz!')) {
        const formData = new FormData();
        formData.append('action', 'delete');
        formData.append('id', id);
        
        fetch('category_actions.php', {
            method: 'POST',
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showNotification('Kategori başarıyla silindi!', 'success');
                setTimeout(() => {
                    window.location.reload();
                }, 1000);
            } else {
                showNotification('Hata: ' + data.message, 'error');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            showNotification('Bir hata oluştu!', 'error');
        });
    }
}

// Bildirim gösterme
function showNotification(message, type = 'info') {
    const alertClass = type === 'success' ? 'alert-success' : 
                      type === 'error' ? 'alert-danger' : 'alert-info';
    
    const alertDiv = document.createElement('div');
    alertDiv.className = `alert ${alertClass} alert-dismissible fade show position-fixed`;
    alertDiv.style.cssText = 'top: 20px; right: 20px; z-index: 9999; min-width: 300px;';
    alertDiv.innerHTML = `
        ${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    `;
    
    document.body.appendChild(alertDiv);
    
    setTimeout(() => {
        if (alertDiv.parentNode) {
            alertDiv.remove();
        }
    }, 5000);
}
</script>

<?php include 'includes/footer.php'; ?>
